# hypersense/importance/random_importance.py
import numpy as np
import pandas as pd
from typing import List, Dict, Any, Optional, Tuple
from hypersense.importance.base_analyzer import BaseImportanceAnalyzer

class RandomImportanceAnalyzer(BaseImportanceAnalyzer):
    """
    Assigns random importance weights to each hyperparameter (normalized to sum=1).
    Used for ablation: removes any signal from the importance module.
    """

    def __init__(self, seed: int = 42):
        super().__init__()
        self.seed = seed
        self.feature_importances_: Optional[Dict[str, float]] = None
        self.param_names_: Optional[List[str]] = None

    def fit(self, configs: List[Dict[str, Any]], scores: List[float]) -> None:
        X = pd.DataFrame(configs)
        self.param_names_ = list(X.columns)

        rng = np.random.default_rng(self.seed)
        weights = rng.random(len(self.param_names_))
        weights = weights / (weights.sum() + 1e-12)

        self.feature_importances_ = {
            name: float(w) for name, w in zip(self.param_names_, weights)
        }

    def explain(self) -> Dict[str, float]:
        if self.feature_importances_ is None:
            raise ValueError("Call fit() before explain().")
        return self.feature_importances_

    def explain_interactions(self) -> Dict[Tuple[str, str], float]:
        return {}

    def rank(self) -> List[str]:
        """
        Return parameter names sorted by importance (desc).
        """
        if self.feature_importances_ is None:
            raise ValueError("Call fit() before rank().")
        return sorted(
            self.feature_importances_.keys(),
            key=lambda k: self.feature_importances_[k],
            reverse=True,
        )
